/*
Licensed to the Apache Software Foundation (ASF) under one or more
contributor license agreements.  See the NOTICE file distributed with
this work for additional information regarding copyright ownership.
The ASF licenses this file to You under the Apache License, Version 2.0
(the "License"); you may not use this file except in compliance with
the License.  You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package kubernetes

import (
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/util/json"

	"github.com/apache/camel-k/pkg/util"
)

// ToJSON marshal to json format.
func ToJSON(value runtime.Object) ([]byte, error) {
	return json.Marshal(value)
}

// ToYAML marshal to yaml format.
func ToYAML(value runtime.Object) ([]byte, error) {
	data, err := ToJSON(value)
	if err != nil {
		return nil, err
	}

	return util.JSONToYAML(data)
}

// ToYAMLNoManagedFields marshal to yaml format but without metadata.managedFields.
func ToYAMLNoManagedFields(value runtime.Object) ([]byte, error) {
	jsondata, err := ToJSON(value)
	if err != nil {
		return nil, err
	}

	mapdata, err := util.JSONToMap(jsondata)
	if err != nil {
		return nil, err
	}

	delete(mapdata["metadata"].(map[string]interface{}), "managedFields")

	return util.MapToYAML(mapdata)
}
