/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.portlet;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletPreferences;
import javax.portlet.PortletURL;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.WindowState;

import org.apache.commons.beanutils.MethodUtils;
import org.apache.commons.collections.keyvalue.DefaultMapEntry;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.ClassUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.portals.bridges.velocity.GenericVelocityPortlet;
import org.apache.portals.messaging.PortletMessaging;
import org.w3c.dom.Element;

/**
 * IFrameGenericPortlet
 * 
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor </a>
 * @version $Id: IFrameGenericPortlet.java 881895 2009-11-18 19:36:33Z woonsan $
 */
public class IFrameGenericPortlet extends GenericVelocityPortlet
{

    public static final String IFRAME_AUTORESIZE_SCRIPT_ID = "org.apache.portals.applications.webcontent.portlet.iframe.autoresize";
    
    public static final String IFRAME_SRC_URL = "org.apache.portals.applications.webcontent.portlet.iframe.src";
    
    public static final String [] SRC_REPLACE_KEYS = { "${serverName}", "${serverPort}", "${contextPath}" };
    
    private Map<String, String> attributes = new HashMap<String, String>();

    private Map<String, String> maxAttributes = new HashMap<String, String>();
    
    private boolean isPortlet10Container;
    
    private String recordVisitResourcePage = "/WEB-INF/view/iframe-record-visit.jsp"; 
    
    @Override
    public void init(PortletConfig config) throws PortletException
    {
        super.init(config);
        
        attributes.put("TITLE", "");
        attributes.put("SRC", "http://www.apache.org");
        attributes.put("PROXYREMOTEURL", "");
        attributes.put("PROXYLOCALPATH", "");
        attributes.put("ALIGN", "BOTTOM");
        attributes.put("CLASS", "");
        attributes.put("FRAMEBORDER", "0");
        attributes.put("ID", "");
        attributes.put("MARGINHEIGHT", "0");
        attributes.put("MARGINWIDTH", "0");
        attributes.put("NAME", "");
        attributes.put("AUTORESIZE", "false");
        attributes.put("VISITLASTPAGE", "false");
        attributes.put("HANDLERSCRIPT", "");

        attributes.put("HEIGHT", "");
        attributes.put("WIDTH", "100%");
        attributes.put("SCROLLING", "NO");
        attributes.put("STYLE", "");

        maxAttributes.put("HEIGHT", "800");
        maxAttributes.put("WIDTH", "100%");
        maxAttributes.put("SCROLLING", "AUTO");
        maxAttributes.put("STYLE", "");
        
        String param = config.getInitParameter("recordVisitResourcePage");
        
        if (param != null)
        {
            recordVisitResourcePage = param;
        }
        
        try
        {
            ClassUtils.getPublicMethod(PortletConfig.class, "getDefaultNamespace", new Class[0]);
        }
        catch (Throwable ignorable)
        {
            isPortlet10Container = true;
        }
    }

    private String getAttributePreference(PortletPreferences prefs, String attribute)
    {
        return this.getMappedAttributePreference(prefs, attribute, attributes);
    }

    private String getMaxAttributePreference(PortletPreferences prefs, String attribute)
    {
        return this.getMappedAttributePreference(prefs, "MAX-" + attribute, maxAttributes);
    }

    private String getMappedAttributePreference(PortletPreferences prefs, String attribute, Map<String, String> map)
    {
        return prefs.getValue(attribute, map.get(attribute));
    }

    private void appendAttribute(PortletPreferences prefs, StringWriter content, String attribute, Map<String, String> map)
    {
        String value;

        if (map == maxAttributes)
            value = getMaxAttributePreference(prefs, attribute);
        else
            value = getAttributePreference(prefs, attribute);

        if (value == null || value.length() == 0) { return; }
        content.append(" ").append(attribute).append("=\"").append(value).append("\"");
    }

    private void appendAttribute(PortletPreferences prefs, StringWriter content, String attribute)
    {
        appendAttribute(prefs, content, attribute, attributes);
    }

    private void appendMaxAttribute(PortletPreferences prefs, StringWriter content, String attribute)
    {
        appendAttribute(prefs, content, attribute, maxAttributes);
    }
    
    protected void doHeaders(RenderRequest request, RenderResponse response) 
    {
        PortletPreferences prefs = request.getPreferences();
        boolean autoResize = BooleanUtils.toBoolean(getAttributePreference(prefs, "AUTORESIZE"));
        boolean visitLastPage = BooleanUtils.toBoolean(getAttributePreference(prefs, "VISITLASTPAGE"));
        
        if (autoResize || visitLastPage)
        {
            String handlerScript = getAttributePreference(prefs, "HANDLERSCRIPT");
            
            if (StringUtils.isBlank(handlerScript))
            {
                handlerScript = request.getContextPath() + "/javascript/iframe_handler.js";
            }
            
            try
            {
                Element headElem = (Element) MethodUtils.invokeMethod(response, "createElement", new Object [] { "script" });
                headElem.setAttribute("id", IFRAME_AUTORESIZE_SCRIPT_ID);
                headElem.setAttribute("language", "JavaScript");
                headElem.setAttribute("type", "text/javascript");
                headElem.setAttribute("src", handlerScript);
                MethodUtils.invokeMethod(response, "addProperty", new Object [] { "javax.portlet.markup.head.element", headElem });
            }
            catch (Exception e)
            {
                throw new RuntimeException("Failed to invoke portlet 2.0 api.", e);
            }
        }
    }
    
    @Override
    public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        String viewPage = (String) request.getAttribute(PARAM_VIEW_PAGE);
        
        if (viewPage != null)
        {
            super.doView(request, response);
        }
        else
        {
            doIFrame(request, response);
        }
    }
    
    @Override
    public void setupPreferencesEdit(RenderRequest request, RenderResponse response)
    {
        PortletPreferences prefs = request.getPreferences();
        
        String editablePrefNames = getAttributePreference(prefs, "EDITABLEPREFS");
        
        if (StringUtils.isBlank(editablePrefNames))
        {
            super.setupPreferencesEdit(request, response);
        }
        else
        {
            List<DefaultMapEntry> prefEntryList = new ArrayList<DefaultMapEntry>();
            Map<String, String> prefsMap = new HashMap<String, String>();
            String [] prefNames = StringUtils.split(editablePrefNames, ",");
            String [] emptyValues = {};
            
            for (String prefName : prefNames)
            {
                prefName = prefName.trim();
                String [] prefValues = prefs.getValues(prefName, emptyValues);
                prefsMap.put(prefName, prefValues.length == 0 ? "" : prefValues[0]);
                prefEntryList.add(new DefaultMapEntry(prefName, prefValues));
            }
            
            getContext(request, response).put("prefs", prefEntryList.iterator());
            getContext(request, response).put("prefsMap", prefsMap);
        }
    }
    
    @Override
    public void doEdit(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        response.setContentType("text/html");
        doPreferencesEdit(request, response);
    }
    
    @Override
    public void doHelp(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        super.doHelp(request, response);
    }
    
    @Override
    public void doCustom(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        super.doCustom(request, response);
    }
    
    /**
     * Render IFRAME content
     */
    protected void doIFrame(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        PortletPreferences prefs = request.getPreferences();
        String source = getURLSource(request, response, prefs);
        // generate HTML IFRAME content
        StringWriter content = new StringWriter(4096);
        
        // fix JS2-349
        content.append("<TABLE CLASS='iframePortletTableContainer' WIDTH='100%'><TBODY CLASS='iframePortletTbodyContainer'><TR><TD>");

        content.append("<IFRAME");

        // special case source
        content.append(" ").append("SRC").append("=\"").append(source).append("\"");

        appendAttribute(prefs, content, "ALIGN");
        appendAttribute(prefs, content, "CLASS");
        appendAttribute(prefs, content, "FRAMEBORDER");
        appendAttribute(prefs, content, "ID");
        appendAttribute(prefs, content, "MARGINHEIGHT");
        appendAttribute(prefs, content, "MARGINWIDTH");
        appendAttribute(prefs, content, "NAME");
        
        if (BooleanUtils.toBoolean(getAttributePreference(prefs, "AUTORESIZE")))
        {
            appendAttribute(prefs, content, "AUTORESIZE");
        }
        
        if (BooleanUtils.toBoolean(getAttributePreference(prefs, "VISITLASTPAGE")))
        {
            appendAttribute(prefs, content, "VISITLASTPAGE");
            
            if (!isPortlet10Container)
            {
                // append a resource url string as an attribute to enable the handler javascript to invoke.
                try
                {
                    Object visitPageResourceURL = MethodUtils.invokeMethod(response, "createResourceURL", null);
                    MethodUtils.invokeMethod(visitPageResourceURL, "setResourceID", new Object [] { recordVisitResourcePage });
                    content.append(" ").append("visitresourceurl").append("=\"");
                    MethodUtils.invokeMethod(visitPageResourceURL, "write", new Object [] { content, Boolean.TRUE });
                    content.append("\"");
                }
                catch (Exception e)
                {
                    throw new PortletException("Failed to invoke portlet 2.0 api.", e);
                }
            }
            else
            {
                PortletURL visitPageActionURL = response.createActionURL();
                visitPageActionURL.setParameter("action", "visit");
                content.append(" ").append("visitresourceurl").append("=\"");
                content.append(visitPageActionURL.toString());
                content.append("\"");
            }
        }
        
        if (request.getWindowState().equals(WindowState.MAXIMIZED))
        {
            appendMaxAttribute(prefs, content, "HEIGHT");
            appendMaxAttribute(prefs, content, "WIDTH");
            appendMaxAttribute(prefs, content, "SCROLLING");
            appendMaxAttribute(prefs, content, "STYLE");
        }
        else
        {
            appendAttribute(prefs, content, "HEIGHT");
            appendAttribute(prefs, content, "WIDTH");
            appendAttribute(prefs, content, "SCROLLING");
            appendAttribute(prefs, content, "STYLE");
        }
        
        content.append(">");
        content.append("<P STYLE=\"textAlign:center\"><A HREF=\"").append(source).append("\">").append(source).append(
                "</A></P>");
        content.append("</IFRAME>");

        // end fix JS2-349
        content.append("</TD></TR></TBODY></TABLE>");
        
        if (isPortlet10Container)
        {
            boolean autoResize = BooleanUtils.toBoolean(getAttributePreference(prefs, "AUTORESIZE"));
            boolean visitLastPage = BooleanUtils.toBoolean(getAttributePreference(prefs, "VISITLASTPAGE"));
            
            if (autoResize || visitLastPage)
            {
                String handlerScript = getAttributePreference(prefs, "HANDLERSCRIPT");
                
                if (StringUtils.isBlank(handlerScript))
                {
                    handlerScript = request.getContextPath() + "/javascript/iframe_handler.js";
                }
                
                content.append("\n<script id=\"").append(IFRAME_AUTORESIZE_SCRIPT_ID).append("\" ");
                content.append("language=\"JavaScript\" type=\"text/javascript\" ");
                content.append("src=\"").append(handlerScript).append("\">");
                content.append("</script>\n");
            }
        }
        
        // set required content type and write HTML IFRAME content
        response.setContentType("text/html");
        response.getWriter().print(content.toString());
    }

    public String getURLSource(RenderRequest request, RenderResponse response, PortletPreferences prefs)
    {
        String [] srcReplaceValues = { request.getServerName(), Integer.toString(request.getServerPort()), request.getContextPath() };
        
        String source = (String) PortletMessaging.receive(request, IFRAME_SRC_URL);
        
        if (source == null)
        {
            source = StringUtils.replaceEach(getAttributePreference(prefs, "SRC"), SRC_REPLACE_KEYS, srcReplaceValues);
        }
        
        // Sometimes, iframe's SRC attribute can be set to a local url to allow cross-domain scripting.
        // If proxy remote URL and its corresponding local path are set, then the proxy remote URL prefix
        // should be replaced by the local path.
        
        String proxyRemoteURL = StringUtils.replaceEach(getAttributePreference(prefs, "PROXYREMOTEURL"), SRC_REPLACE_KEYS, srcReplaceValues);
        String proxyLocalPath = StringUtils.replaceEach(getAttributePreference(prefs, "PROXYLOCALPATH"), SRC_REPLACE_KEYS, srcReplaceValues);
        
        if (StringUtils.isNotEmpty(proxyRemoteURL) && StringUtils.isNotEmpty(proxyLocalPath) && StringUtils.startsWith(source, proxyRemoteURL))
        {
            source = proxyLocalPath + source.substring(proxyRemoteURL.length());
        }
        
        return source;
    }

    /**
     * Save the prefs
     */
    @Override
    public void processAction(ActionRequest request, ActionResponse actionResponse) throws PortletException,
            IOException
    {
        if ("visit".equals(request.getParameter("action")))
        {
            String url = request.getParameter("URL");
            
            if (!StringUtils.isBlank(url))
            {
                PortletMessaging.publish(request, IFRAME_SRC_URL, url.trim());
            }
        }
        else
        {
            processPreferencesAction(request, actionResponse);
        }
    }
    
    @Override
    protected String getTitle(RenderRequest request) 
    {
        String title = getAttributePreference(request.getPreferences(), "TITLE");
        
        if (!StringUtils.isEmpty(title))
        {
            return title;
        }
        else
        {
            return super.getTitle(request);
        }
    }
    
}
