/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.rewriter;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;

import org.apache.commons.io.output.TeeOutputStream;
import org.apache.commons.lang.StringUtils;
import org.apache.portals.applications.webcontent2.rewriter.impl.AbstractTextLineContentRewriter;
import org.apache.portals.applications.webcontent2.rewriter.impl.SimpleContentRewritingContext;
import org.junit.Before;
import org.junit.Test;

public class TextLineContentRewriterTest
{

    private Source source;
    private Sink sink;
    private ByteArrayOutputStream sinkTeeOut;
    private ContentRewriter contentRewriter;

    @Before
    public void before() {
        source = new Source() {
            public InputStream getInputStream() throws IOException
            {
                return TextLineContentRewriterTest.class.getResourceAsStream("announce-template.txt");
            }
            public Reader getReader() throws IOException
            {
                return new InputStreamReader(getInputStream());
            }
        };

        sinkTeeOut = new ByteArrayOutputStream();
        sink = new Sink() {
            public OutputStream getOutputStream() throws IOException
            {
                return new TeeOutputStream(System.out, sinkTeeOut);
            }
            public Writer getWriter() throws IOException
            {
                return new OutputStreamWriter(getOutputStream());
            }
        };

        contentRewriter = new AbstractTextLineContentRewriter() {
            @Override
            protected String rewriteLine(String line, ContentRewritingContext context) throws ContentRewritingException, IOException
            {
                return StringUtils.replace(line, "1.3", "2.0");
            }
        };

    }

    @Test
    public void testRewriter() throws Exception {
        ContentRewritingContext rewritingContext = new SimpleContentRewritingContext();
        contentRewriter.rewrite(source, sink, rewritingContext);
        String output = sinkTeeOut.toString();
        assertFalse(output.contains("[ANNOUNCE] Apache Portals Applications WebContent version 1.3 Release"));
        assertTrue(output.contains("[ANNOUNCE] Apache Portals Applications WebContent version 2.0 Release"));
    }

}
