/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.portlet.history;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;

/**
 * History of web content page navigations
 */
public class WebContentPageHistory implements Serializable
{
    private static final long serialVersionUID = 1L;

    private List<WebContentPage> pageList;
    private int curIndex = -1;

    public WebContentPageHistory()
    {
        pageList = new LinkedList<WebContentPage>();
    }

    public boolean isEmpty()
    {
        return (pageList == null || pageList.isEmpty());
    }

    public boolean hasCurrentPage()
    {
        return curIndex >= 0;
    }

    public boolean hasPreviousPage()
    {
        return !isEmpty() && (curIndex > 0);
    }

    public boolean hasNextPage()
    {
        return !isEmpty() && (curIndex < pageList.size() - 1);
    }

    public WebContentPage getCurrentPage()
    {
        if (!hasCurrentPage())
        {
            return null;
        }

        return (WebContentPage) pageList.get(curIndex);
    }

    public WebContentPage getPreviousPage()
    {
        if (!hasPreviousPage())
        {
            return null;
        }

        curIndex -= 1;

        return getCurrentPage();
    }

    public WebContentPage getNextPage()
    {
        if (!hasNextPage())
        {
            return null;
        }

        curIndex += 1;

        return getCurrentPage();
    }

    public void visitPage(WebContentPage page)
    {
        if (page == null)
        {
            throw new IllegalArgumentException("page must be not null.");
        }

        int index = pageList.indexOf(page);

        if (index != -1 && index == curIndex)
        {
            // just you're visiting the current page
            return;
        }

        // otherwise new page...
        while (hasNextPage())
        {
            // ...visiting a page discards any pages we have visited by going "back"
            pageList.remove(curIndex + 1);
        }

        if (index != -1 && index < pageList.size())
        {
            // ...actually, new visit to an old page, only keep one reference to it
            pageList.remove(index);
        }

        // add in the new page, at the end
        pageList.add(page);
        curIndex = pageList.size() - 1;
    }
}
