/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.portlet.proxy;

import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import javax.portlet.MimeResponse;
import javax.portlet.PortletResponse;
import javax.portlet.PortletURL;

import org.apache.http.client.utils.URIUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.util.RewriterUtils;
import org.apache.portals.applications.webcontent2.rewriter.htmlcleaner.AbstractProxyTagNodeVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultPortletProxyTagNodeVisitor extends AbstractProxyTagNodeVisitor
{

    private static Logger log = LoggerFactory.getLogger(DefaultPortletProxyTagNodeVisitor.class);

    private static final Map<String, String[]> DEFAULT_TAG_NAME_AND_LINK_ATTRS = new HashMap<String, String[]>();

    static
    {
        DEFAULT_TAG_NAME_AND_LINK_ATTRS.put("a", new String[] { "href" });
        DEFAULT_TAG_NAME_AND_LINK_ATTRS.put("img", new String[] { "src" });
        DEFAULT_TAG_NAME_AND_LINK_ATTRS.put("form", new String[] { "action" });
    }

    public DefaultPortletProxyTagNodeVisitor()
    {
        this(DEFAULT_TAG_NAME_AND_LINK_ATTRS);
    }

    public DefaultPortletProxyTagNodeVisitor(Map<String, String[]> tagNameAndLinkAttrs)
    {
        super(tagNameAndLinkAttrs);
    }

    @Override
    protected String rewriteURI(String tagName, String attrName, String uri)
    {
        if (!isRewritableURI(uri))
        {
            return uri;
        }

        ProxyContext proxyContext = ProxyContext.getCurrentProxyContext();
        PortletRequestContext prc = (PortletRequestContext) proxyContext.getRequestContext();
        PortletResponse response = prc.getPortletResponse();

        if (!(response instanceof MimeResponse))
        {
            log.error("Cannot rewrite url because response is not a MimeResponse.");
            return uri;
        }

        URI remoteURI = proxyContext.getRemoteURI();

        URI uriObj = null;

        try
        {
            uriObj = URI.create(uri);
        }
        catch (Exception e)
        {
            log.warn("Invalid uri: '{}'.", uri);
            return uri;
        }

        URI rewrittenURI = URIUtils.resolve(remoteURI, uriObj);

        if ("src".equals(attrName))
        {
            return rewrittenURI.toString();
        }
        else
        {
            PortletURL actionUrl = ((MimeResponse) response).createActionURL();
            actionUrl.setParameter(GenericReverseProxyPortlet.REMOTE_URI_PARAM_NAME, rewrittenURI.toString());
            return actionUrl.toString();
        }
    }

    protected boolean isRewritableURI(String uri)
    {
        return RewriterUtils.isRewritableURI(uri);
    }
}
