/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.portlet.rewriter;

import static org.junit.Assert.assertNotNull;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;

import org.apache.portals.applications.webcontent2.portlet.rewriter.html.neko.NekoParserAdaptor;
import org.apache.portals.applications.webcontent2.portlet.rewriter.rules.Ruleset;
import org.apache.portals.applications.webcontent2.portlet.rewriter.xml.SaxParserAdaptor;
import org.junit.Test;


/**
 * TestNekoRewriter
 * 
 * @author <a href="mailto:dyoung@phase2systems.com">David L Young</a>
 * @version $Id: TestNekoRewriter.java 771214 2009-05-04 07:08:16Z vkumar $
 */
public class TestNekoRewriter extends AbstractRewriterTestCase
{

    // NekoParserAdapter test
    @Test
    public void testNekoParserAdaptor() throws Exception
    {
        RewriterController controller = getController();
        FileReader rulesReader = getTestReader("test-remove-rules.xml");
        Ruleset ruleset = controller.loadRuleset(rulesReader);
        rulesReader.close();
        assertNotNull("ruleset is null", ruleset);
        RulesetRewriter rewriter = controller.createRewriter(ruleset);
        assertNotNull("ruleset rewriter is null", rewriter);

        FileReader htmlReader = getTestReader("test-001.html");
        FileWriter htmlWriter = getTestWriter("test-002-output.html");

        ParserAdaptor adaptor = controller.createParserAdaptor("text/html");
        rewriter.setBaseUrl("http://www.rewriter.com");
        rewriter.rewrite(adaptor, htmlReader, htmlWriter);
        htmlReader.close();
    }
    
    private RewriterController getController() throws Exception
    {
        Class[] rewriterClasses = new Class[]{ RulesetRewriterImpl.class, RulesetRewriterImpl.class};
        Class[] adaptorClasses = new Class[]{ NekoParserAdaptor.class, SaxParserAdaptor.class};
        return new MappingRewriterController(getBaseDir()+"src/test/webapp/WEB-INF/conf/rewriter-rules-mapping.xml", Arrays.asList(rewriterClasses), Arrays.asList(adaptorClasses));
    }

    /**
     * Gets a reader for a given filename in the test directory.
     * 
     * @return A file reader to the test rules file
     * @throws IOException
     */
    private FileReader getTestReader(String filename) throws IOException
    {
        return new FileReader(getBaseDir() + "src/test/rewriter/" + filename);
    }

    /**
     * Gets a writer for a given filename in the test directory.
     * 
     * @return A file reader to the test rules file
     * @throws IOException
     */
    private FileWriter getTestWriter(String filename) throws IOException
    {
        new File(getBaseDir()+"target/test/rewriter").mkdirs();
        return new FileWriter(getBaseDir()+"target/test/rewriter/" + filename);
    }
}
