/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy;

import java.net.URI;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.chain.impl.ContextBase;
import org.apache.http.HttpResponse;
import org.apache.http.client.CookieStore;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewriter;

/**
 * Base {@link org.apache.commons.chain.Context} implementation to be used in
 * a reverse proxy service execution.
 * @see {@link org.apache.commons.chain.Context}
 */
public class ProxyContext extends ContextBase
{

    private static final long serialVersionUID = 1L;

    /**
     * Static thread local to store/retrieve the current <code>ProxyContext</code> instance.
     * In some cases, a component, during the reverse proxy processing, might not have an access
     * to the <code>ProxyContext</code> instance. In that case, that kind of component might
     * want to invoke {@link #getCurrentProxyContext()} to retrieve the current <code>ProxyContext</code>
     * instance.
     */
    private static ThreadLocal<ProxyContext> tlProxyContext = new ThreadLocal<ProxyContext>();

    /**
     * Returns the current <code>ProxyContext</code> instance in the static thread local variable.
     * In some cases, a component, during the reverse proxy processing, might not have an access
     * to the <code>ProxyContext</code> instance. In that case, that kind of component might
     * want to invoke this method to retrieve the current <code>ProxyContext</code> instance.
     * @return
     */
    public static ProxyContext getCurrentProxyContext()
    {
        return tlProxyContext.get();
    }

    /**
     * Sets the current <code>ProxyContext</code> instance in the static thread local variable.
     * @param proxyContext
     */
    public static void setCurrentProxyContext(ProxyContext proxyContext)
    {
        tlProxyContext.set(proxyContext);
    }

    /**
     * Removes the current <code>ProxyContext</code> instance from the static thread local variable.
     */
    public static void removeCurrentProxyContext()
    {
        tlProxyContext.remove();
    }

    /**
     * HTTP request related context information.
     * For example, in a normal servlet runtime, it may contain servlet request related attributes,
     * whereas it may contain portlet related attribute in a portlet runtime.
     */
    private final RequestContext requestContext;

    /**
     * The reverse proxy mapping registry used in the current reverse proxy processing.
     */
    private ProxyMappingRegistry proxyMappingRegistry;

    /**
     * The resolved reverse proxy mapping instance for the current reverse proxy processing.
     */
    private ProxyMapping resolvedMapping;

    /**
     * The local path info of the current request from a client or caller.
     */
    private String localPath;

    /**
     * The resolved remote target URI for the current reverse proxy processing.
     */
    private URI remoteURI;

    /**
     * <code>HttpClient</code> instance used in the current reverse proxy processing.
     */
    private HttpClient httpClient;

    /**
     * <code>CookieStore</code> to store all the HTTP cookie data in the current reverse proxy processing.
     */
    private CookieStore cookieStore;

    /**
     * {@link HttpRequestBase} instance used during the current reverse proxy processing.
     */
    private HttpRequestBase httpRequest;

    /**
     * {@link HttpResponse} instance used during the current reverse proxy processing.
     */
    private HttpResponse httpResponse;

    /**
     * {@link ContentRewriter} instance used to rewrite the response content if any set.
     */
    private ContentRewriter contentRewriter;

    /**
     * List of exceptions during the reverse proxy processing if there's any.
     */
    private List<Exception> exceptions;

    /**
     * Constructs a <code>ProxyContext</code> for the given <code>RequestContext</code>.
     * @param requestContext
     */
    public ProxyContext(final RequestContext requestContext) 
    {
        super();
        this.requestContext = requestContext;
    }

    /**
     * Constructs a <code>ProxyContext</code> for the given <code>RequestContext</code>
     * with putting all entries into the <code>ProxyContext</code> from the initial <code>map</code>.
     * @param requestContext
     * @param map
     */
    public ProxyContext(final RequestContext requestContext, @SuppressWarnings("rawtypes") Map map) 
    {
        super(map);
        this.requestContext = requestContext;
    }

    /**
     * Returns the {@link RequestContext} instance in this reverse proxy processing context.
     * @return
     */
    public RequestContext getRequestContext()
    {
        return requestContext;
    }

    /**
     * Returns the reverse proxy mapping registry in this reverse proxy processing context.
     * @return
     */
    public ProxyMappingRegistry getProxyMappingRegistry()
    {
        return proxyMappingRegistry;
    }

    /**
     * Sets the reverse proxy mapping registry in this reverse proxy processing context.
     * @param proxyMappingRegistry
     */
    public void setProxyMappingRegistry(ProxyMappingRegistry proxyMappingRegistry)
    {
        this.proxyMappingRegistry = proxyMappingRegistry;
    }

    /**
     * Returns the resolved reverse proxy mapping in this reverse proxy processing context.
     * @return
     */
    public ProxyMapping getResolvedMapping()
    {
        return resolvedMapping;
    }

    /**
     * Sets the resolved reverse proxy mapping in this reverse proxy processing context.
     * @param resolvedMapping
     */
    public void setResolvedMapping(ProxyMapping resolvedMapping)
    {
        this.resolvedMapping = resolvedMapping;
    }

    /**
     * Returns the local path info in this reverse proxy processing context.
     * @return
     */
    public String getLocalPath()
    {
        return localPath;
    }

    /**
     * Sets the local path info in this reverse proxy processing context.
     * @param localPath
     */
    public void setLocalPath(String localPath)
    {
        this.localPath = localPath;
    }

    /**
     * Returns the remote target URI in this reverse proxy processing context.
     * @return
     */
    public URI getRemoteURI()
    {
        return remoteURI;
    }

    /**
     * Sets the remote target URI in this reverse proxy processing context.
     * @param remoteURI
     */
    public void setRemoteURI(URI remoteURI)
    {
        this.remoteURI = remoteURI;
    }

    /**
     * Returns the internal <code>HttpClient</code> instance used in this reverse proxy processing context.
     * @return
     */
    public HttpClient getHttpClient()
    {
        return httpClient;
    }

    /**
     * Sets the internal <code>HttpClient</code> instance to be used in this reverse proxy processing context.
     * @param httpClient
     */
    public void setHttpClient(HttpClient httpClient)
    {
        this.httpClient = httpClient;
    }

    /**
     * Returns the {@link CookieStore} instance used in this reverse proxy processing context.
     * @return
     */
    public CookieStore getCookieStore()
    {
        return cookieStore;
    }

    /**
     * Sets the {@link CookieStore} instance to be used in this reverse proxy processing context.
     * @param cookieStore
     */
    public void setCookieStore(CookieStore cookieStore)
    {
        this.cookieStore = cookieStore;
    }

    /**
     * Returns the {@link HttpRequestBase} instance used in the current reverse proxy processing context.
     * @return
     */
    public HttpRequestBase getHttpRequest()
    {
        return httpRequest;
    }

    /**
     * Sets the {@link HttpRequestBase} instance to be used in the current reverse proxy processing context.
     * @param httpRequest
     */
    public void setHttpRequest(HttpRequestBase httpRequest)
    {
        this.httpRequest = httpRequest;
    }

    /**
     * Returns the {@link HttpResponse} instance used in the current reverse proxy processing context.
     * @return
     */
    public HttpResponse getHttpResponse()
    {
        return httpResponse;
    }

    /**
     * Sets the {@link HttpResponse} instance to be used in the current reverse proxy processing context.
     * @param httpResponse
     */
    public void setHttpResponse(HttpResponse httpResponse)
    {
        this.httpResponse = httpResponse;
    }

    /**
     * Returns the {@link ContentRewriter} instance used in the current reverse proxy processing context.
     * @return
     */
    public ContentRewriter getContentRewriter()
    {
        return contentRewriter;
    }

    /**
     * Sets the {@link ContentRewriter} instance to be used in the current reverse proxy processing context.
     * @param contentRewriter
     */
    public void setContentRewriter(ContentRewriter contentRewriter)
    {
        this.contentRewriter = contentRewriter;
    }

    /**
     * Returns true if there's any exception caught during the current reverse proxy processing.
     * Otherwise returns false.
     * @return
     */
    public boolean hasException()
    {
        return (exceptions != null && !exceptions.isEmpty());
    }

    /**
     * Adds an exception caught during the current reverse proxy processing.
     * @param e
     */
    public void addException(Exception e)
    {
        if (exceptions == null)
        {
            exceptions = new LinkedList<Exception>();
        }

        exceptions.add(e);
    }

    /**
     * Returns a list of exception(s) caught during the current reverse proxy processing.
     * @return
     */
    public List<Exception> getExceptions()
    {
        return exceptions;
    }
}
