/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.builder;

import org.apache.commons.chain.Chain;
import org.apache.commons.chain.Command;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.portals.applications.webcontent2.proxy.HttpClientContextBuilder;
import org.apache.portals.applications.webcontent2.proxy.ProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.command.AddCookiesToResponseCommand;
import org.apache.portals.applications.webcontent2.proxy.command.AddHeaderToHttpRequestCommand;
import org.apache.portals.applications.webcontent2.proxy.command.AddHeadersToResponseCommand;
import org.apache.portals.applications.webcontent2.proxy.command.CleanupCommand;
import org.apache.portals.applications.webcontent2.proxy.command.ExecuteHttpClientCommand;
import org.apache.portals.applications.webcontent2.proxy.command.HandleNotModifiedCommand;
import org.apache.portals.applications.webcontent2.proxy.command.HandleRedirectionCommand;
import org.apache.portals.applications.webcontent2.proxy.command.InitHttpClientCommand;
import org.apache.portals.applications.webcontent2.proxy.command.InitHttpRequestCommand;
import org.apache.portals.applications.webcontent2.proxy.command.InitializationCommand;
import org.apache.portals.applications.webcontent2.proxy.command.ResolveContentRewriterCommand;
import org.apache.portals.applications.webcontent2.proxy.command.ResolveLocalPathCommand;
import org.apache.portals.applications.webcontent2.proxy.command.ResolveProxyMappingCommand;
import org.apache.portals.applications.webcontent2.proxy.command.ResolveRemoteURICommand;
import org.apache.portals.applications.webcontent2.proxy.command.SerializeHttpEntityContentCommand;
import org.apache.portals.applications.webcontent2.proxy.impl.DefaultURICleaner;
import org.apache.portals.applications.webcontent2.proxy.impl.ProxyProcessingChain;

/**
 * Default {@link ProxyProcessingChainBuilder} implementation.
 */
public class DefaultProxyProcessingChainBuilder extends AbstractProxyProcessingChainBuilder
{

    /**
     * Zero-argument default constructor.
     */
    public DefaultProxyProcessingChainBuilder()
    {
        super();
    }

    /**
     * {@inheritDoc}
     * <p>
     * The implementation adds three {@link Chain} commands for each reverse proxy processing phase.
     * That is, <em>pre-processing phase</em>, <em>processing phase</em> and <em>post-processing phase</em>.
     * </p>
     * <p>
     * In <em>pre-processing phase</em>, all the initialization related {@link Command}s are added.
     * In <em>processing phase</em>, core reverse proxy processing {@link Command}s are added.
     * In <em>post-processing phase</em>, all the resource cleanup related {@link Command}s are added.
     * </p>
     * <p>
     * The {@link Chain} for <em>pre-processing phase</em> is created by {@link #createDefaultPreprocessingCommand(ProxyMappingRegistry, HttpClientBuilder, HttpClientContextBuilder)}.
     * The {@link Chain} for <em>processing phase</em> is created by {@link #createDefaultProcessingCommand(ProxyMappingRegistry, HttpClientBuilder, HttpClientContextBuilder)}.
     * Finally the {@link Chain} for <em>post-processing phase</em> is created by {@link #createDefaultPostprocessingCommand(ProxyMappingRegistry, HttpClientBuilder, HttpClientContextBuilder)}.
     * </p>
     */
    @Override
    protected void initializeCommands(ProxyMappingRegistry proxyMappingRegistry, HttpClientBuilder httpClientBuilder,
                                                HttpClientContextBuilder httpClientContextBuilder)
    {
        addCommand(createDefaultPreprocessingCommand(proxyMappingRegistry, httpClientBuilder, httpClientContextBuilder));
        addCommand(createDefaultProcessingCommand(proxyMappingRegistry, httpClientBuilder, httpClientContextBuilder));
        addCommand(createDefaultPostprocessingCommand(proxyMappingRegistry, httpClientBuilder, httpClientContextBuilder));
    }

    /**
     * Creates a {@link ProxyProcessingChain} which contains all the {@link Command}s in the pre-processing phase.
     * @param proxyMappingRegistry
     * @param httpClientBuilder
     * @param httpClientContextBuilder
     * @return
     */
    protected ProxyProcessingChain createDefaultPreprocessingCommand(ProxyMappingRegistry proxyMappingRegistry, HttpClientBuilder httpClientBuilder, HttpClientContextBuilder httpClientContextBuilder)
    {
        ProxyProcessingChain preprocessingChain = new ProxyProcessingChain();
        InitializationCommand initializationCommand = new InitializationCommand();
        preprocessingChain.addCommand(initializationCommand);
        return preprocessingChain;
    }

    /**
     * Creates a {@link ProxyProcessingChain} which contains all the {@link Command}s in the core processing phase.
     * @param proxyMappingRegistry
     * @param httpClientBuilder
     * @param httpClientContextBuilder
     * @return
     */
    protected ProxyProcessingChain createDefaultProcessingCommand(ProxyMappingRegistry proxyMappingRegistry, HttpClientBuilder httpClientBuilder, HttpClientContextBuilder httpClientContextBuilder)
    {
        ProxyProcessingChain processingChain = new ProxyProcessingChain();

        ResolveLocalPathCommand resolveLocalPathCommand = new ResolveLocalPathCommand();

        ResolveProxyMappingCommand resolveProxyMappingCommand = new ResolveProxyMappingCommand(proxyMappingRegistry);

        ResolveRemoteURICommand resolveRemoteURICommand = new ResolveRemoteURICommand();
        resolveRemoteURICommand.setUriCleaner(new DefaultURICleaner());

        InitHttpClientCommand initHttpClientCommand = new InitHttpClientCommand();
        initHttpClientCommand.setHttpClientBuilder(httpClientBuilder);

        InitHttpRequestCommand initHttpRequestCommand = new InitHttpRequestCommand();
        AddHeaderToHttpRequestCommand addHeaderToHttpRequestCommand = new AddHeaderToHttpRequestCommand();

        ExecuteHttpClientCommand executeHttpClientCommand = new ExecuteHttpClientCommand();
        executeHttpClientCommand.setHttpClientContextBuilder(httpClientContextBuilder);

        ResolveContentRewriterCommand resolveContentRewriterCommand = new ResolveContentRewriterCommand();
        AddHeadersToResponseCommand addHeadersToResponseCommand = new AddHeadersToResponseCommand();
        AddCookiesToResponseCommand addCookiesToResponseCommand = new AddCookiesToResponseCommand();
        HandleRedirectionCommand handleRedirectionCommand = new HandleRedirectionCommand();
        HandleNotModifiedCommand handleNotModifiedCommand = new HandleNotModifiedCommand();
        SerializeHttpEntityContentCommand serializeHttpEntityContentCommand = new SerializeHttpEntityContentCommand();

        processingChain.addCommand(resolveLocalPathCommand);
        processingChain.addCommand(resolveProxyMappingCommand);
        processingChain.addCommand(resolveRemoteURICommand);
        processingChain.addCommand(initHttpClientCommand);
        processingChain.addCommand(initHttpRequestCommand);
        processingChain.addCommand(addHeaderToHttpRequestCommand);
        processingChain.addCommand(executeHttpClientCommand);
        processingChain.addCommand(resolveContentRewriterCommand);
        processingChain.addCommand(addHeadersToResponseCommand);
        processingChain.addCommand(addCookiesToResponseCommand);
        processingChain.addCommand(handleRedirectionCommand);
        processingChain.addCommand(handleNotModifiedCommand);
        processingChain.addCommand(serializeHttpEntityContentCommand);

        return processingChain;
    }

    /**
     * Creates a {@link ProxyProcessingChain} which contains all the {@link Command}s in the post-processing phase.
     * @param proxyMappingRegistry
     * @param httpClientBuilder
     * @param httpClientContextBuilder
     * @return
     */
    protected ProxyProcessingChain createDefaultPostprocessingCommand(ProxyMappingRegistry proxyMappingRegistry, HttpClientBuilder httpClientBuilder, HttpClientContextBuilder httpClientContextBuilder)
    {
        ProxyProcessingChain postprocessingChain = new ProxyProcessingChain();
        CleanupCommand cleanupCommand = new CleanupCommand();
        postprocessingChain.addCommand(cleanupCommand);
        return postprocessingChain;
    }
}
