/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.chain.Command;
import org.apache.http.HttpResponse;
import org.apache.http.protocol.HTTP;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;


/**
 * {@link Command} responsible for handling HTTP 304 error response from the remote target content.
 */
public class HandleNotModifiedCommand extends AbstractProxyCommand
{

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpResponse httpResponse = context.getHttpResponse();

        int statusCode = httpResponse.getStatusLine().getStatusCode();

        if (statusCode != HttpServletResponse.SC_NOT_MODIFIED)
        {
            return false;
        }

        // 304 needs special handling. See:
        // http://www.ics.uci.edu/pub/ietf/http/rfc1945.html#Code304
        // We get a 304 whenever passed an 'If-Modified-Since'
        // header and the data on disk has not changed; server
        // responds w/ a 304 saying I'm not going to send the
        // body because the file has not changed.
        context.getRequestContext().setIntHeader(HTTP.CONTENT_LEN, 0);
        context.getRequestContext().setStatus(HttpServletResponse.SC_NOT_MODIFIED);

        return true;
    }

}
