/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.zip.GZIPInputStream;

import org.apache.commons.chain.Command;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.entity.ContentType;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.ProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;
import org.apache.portals.applications.webcontent2.proxy.impl.GzippedSource;
import org.apache.portals.applications.webcontent2.proxy.impl.HttpEntitySource;
import org.apache.portals.applications.webcontent2.proxy.util.RequestUtils;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewriter;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewritingContext;
import org.apache.portals.applications.webcontent2.rewriter.Sink;
import org.apache.portals.applications.webcontent2.rewriter.Source;
import org.apache.portals.applications.webcontent2.rewriter.impl.SimpleContentRewritingContext;

/**
 * {@link Command} responsible for serializing the response body from the remote content
 * to the response in the reverse proxy side.
 */
public class SerializeHttpEntityContentCommand extends AbstractProxyCommand
{

    /**
     * Default character encoding to be used when serializing content to response.
     */
    private String defaultCharacterEncoding = "UTF-8";

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpResponse httpResponse = context.getHttpResponse();
        HttpEntity httpEntity = httpResponse.getEntity();

        if (httpEntity != null)
        {
            if (RequestUtils.isDispatched(context.getRequestContext()))
            {
                writeHttpEntityToDispatcher(context, httpEntity);
            }
            else
            {
                writeHttpEntityToResponse(context, httpEntity);
            }
        }

        return false;
    }

    /**
     * Returns the default character encoding to be used when serializing content to response.
     * @return
     */
    public String getDefaultCharacterEncoding()
    {
        return defaultCharacterEncoding;
    }

    /**
     * Sets the default character encoding to be used when serializing content to response.
     * @param defaultCharacterEncoding
     */
    public void setDefaultCharacterEncoding(String defaultCharacterEncoding)
    {
        this.defaultCharacterEncoding = defaultCharacterEncoding;
    }

    /**
     * Write the {@link HttpEntity} of {@link HttpResponse} from the remote content
     * to the non-dispatched response in the reverse proxy side.
     * <p>
     * If a {@link ContentRewriter} was resolved in the previous step, then
     * it used the {@link ContentRewriter} to rewriter the response content.
     * </p>
     * @param context
     * @param httpEntity
     * @throws ReverseProxyException
     * @throws IOException
     */
    protected void writeHttpEntityToResponse(final ProxyContext context, final HttpEntity httpEntity) throws ReverseProxyException, IOException
    {
        Source source = new HttpEntitySource(httpEntity);
        Sink sink = context.getRequestContext().getSink();

        ContentType contentType = ContentType.getOrDefault(httpEntity);
        Charset charset = contentType.getCharset();
        String charsetName = StringUtils.defaultIfEmpty((charset != null ? charset.name() : null), getDefaultCharacterEncoding());

        boolean gzipEncoded = isGzipEncodedContent(httpEntity);

        ContentRewriter contentRewriter = context.getContentRewriter();

        if (contentRewriter == null)
        {
            InputStream in = null;
            OutputStream out = null;

            try
            {
                in = source.getInputStream();
                // According to javadoc of httpclient, getResponseBodyAsStream() can return null
                // if the response has no body.
                if (in != null)
                {
                    out = sink.getOutputStream();
                    IOUtils.copy(in, out);
                    out.flush();
                }
            }
            finally
            {
                if (in != null)
                {
                    IOUtils.closeQuietly(in);
                }

                if (out != null)
                {
                    IOUtils.closeQuietly(out);
                }
            }
        }
        else
        {
            ContentRewritingContext rewritingContext = createContentRewritingContext(context);

            if (gzipEncoded)
            {
                contentRewriter.rewrite(new GzippedSource(source, charsetName), sink, rewritingContext);
            }
            else
            {
                contentRewriter.rewrite(source, sink, rewritingContext);
            }
        }
    }

    /**
     * Write the {@link HttpEntity} of {@link HttpResponse} from the remote content
     * to the dispatched response in the reverse proxy side.
     * <p>
     * If a {@link ContentRewriter} was resolved in the previous step, then
     * it used the {@link ContentRewriter} to rewriter the response content.
     * </p>
     * @param context
     * @param httpEntity
     * @throws ReverseProxyException
     * @throws IOException
     */
    protected void writeHttpEntityToDispatcher(final ProxyContext context, final HttpEntity httpEntity) throws ReverseProxyException, IOException
    {
        Source source = new HttpEntitySource(httpEntity);
        Sink sink = context.getRequestContext().getSink();

        ContentType contentType = ContentType.getOrDefault(httpEntity);
        Charset charset = contentType.getCharset();
        String charsetName = StringUtils.defaultIfEmpty((charset != null ? charset.name() : null), getDefaultCharacterEncoding());

        boolean gzipEncoded = isGzipEncodedContent(httpEntity);

        ContentRewriter contentRewriter = context.getContentRewriter();

        InputStream in = null;
        Reader reader = null;
        OutputStream out = null;
        Writer writer = null;

        try
        {
            if (contentRewriter != null)
            {
                ContentRewritingContext rewritingContext = createContentRewritingContext(context);

                if (gzipEncoded)
                {
                    contentRewriter.rewrite(new GzippedSource(source, charsetName), sink, rewritingContext);
                }
                else
                {
                    contentRewriter.rewrite(source, sink, rewritingContext);
                }
            }
            else
            {
                in = source.getInputStream();

                // According to javadoc of httpclient, getResponseBodyAsStream() can return null
                // if the response has no body.
                if (in != null)
                {
                    try
                    {
                        out = sink.getOutputStream();
                    }
                    catch (IllegalStateException e)
                    {
                        writer = sink.getWriter();
                    }

                    if (out != null)
                    {
                        IOUtils.copy(in, out);
                        out.flush();
                    }
                    else
                    {
                        if (charsetName != null)
                        {
                            reader = new InputStreamReader(gzipEncoded ? new GZIPInputStream(in) : in, charsetName);
                        }
                        else
                        {
                            reader = new InputStreamReader(gzipEncoded ? new GZIPInputStream(in) : in);
                        }

                        IOUtils.copy(reader, writer);
                        writer.flush();
                    }
                }
            }
        }
        finally
        {
            if (reader != null)
            {
                IOUtils.closeQuietly(reader);
            }

            if (in != null)
            {
                IOUtils.closeQuietly(in);
            }
        }
    }

    /**
     * Creates a new {@link ContentRewritingContext} which is to be used by the resolved {@link ContentRewriter}.
     * @param proxyContext
     * @return
     */
    protected ContentRewritingContext createContentRewritingContext(final ProxyContext proxyContext)
    {
        SimpleContentRewritingContext contentRewritingContext = new SimpleContentRewritingContext();

        contentRewritingContext.setAttribute(ProxyContext.class.getName(), proxyContext);
        contentRewritingContext.setAttribute(ProxyMapping.class.getName(), proxyContext.getResolvedMapping());
        contentRewritingContext.setAttribute(ProxyMappingRegistry.class.getName(), proxyContext.getProxyMappingRegistry());

        return contentRewritingContext;
    }

    /**
     * Returns true if the <code>Content-Encoding</code> HTTP response header from the remote content is "gzip".
     * Otherwise, it returns false.
     * @param httpEntity
     * @return
     */
    private boolean isGzipEncodedContent(final HttpEntity httpEntity) 
    {
        boolean gzipEncoded = false;
        Header contentEncodingHeader = httpEntity.getContentEncoding();

        if (contentEncodingHeader != null)
        {
            gzipEncoded = StringUtils.equalsIgnoreCase("gzip", contentEncodingHeader.getValue());
        }

        return gzipEncoded;
    }

}
