/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.io.IOException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.chain.Chain;
import org.apache.commons.chain.Command;
import org.apache.commons.chain.Context;
import org.apache.commons.chain.impl.ChainBase;
import org.apache.commons.lang.ArrayUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;

/**
 * <p>Convenience base class for reverse proxy processing {@link Chain} implementation.</p>
 */
public class ProxyProcessingChain extends ChainBase
{

    /**
     * Constructs a <code>ProxyProcessingChain</code> with no configured {@link Command}s.</p>
     */
    public ProxyProcessingChain() 
    {
        super();
    }

    /**
     * Constructs a <code>ProxyProcessingChain</code> configured with the specified
     * {@link Command}.</p>
     * @param command
     */
    public ProxyProcessingChain(Command command) 
    {
        super(command);
    }

    /**
     * Constructs a <code>ProxyProcessingChain</code> configured with the specified
     * {@link Command}s.</p>
     * @param commands
     */
    public ProxyProcessingChain(Command[] commands) 
    {
        super(commands);
    }

    /**
     * Constructs a <code>ProxyProcessingChain</code> configured with the specified
     * {@link Command}s.</p>
     * @param commands
     */
    public ProxyProcessingChain(@SuppressWarnings("rawtypes") Collection commands) 
    {
        super(commands);
    }

    /**
     * Return the size of the internal {@link Command} list.
     * @return
     */
    public int getCommandCount() {
        return commands.length;
    }

    /**
     * Return the index of the given {@link Command} in the command list.
     * If not found, it should return -1.
     * @param command
     * @return
     */
    public int getCommandIndex(Command command) {
        return ArrayUtils.indexOf(commands, command);
    }

    /**
     * Return the index of the given class type of {@link Command} in the command list.
     * If not found, it should return -1.
     * @param commandType
     * @return
     */
    public int getCommandIndex(Class<? extends Command> commandType) {
        if (commands == null) {
            return -1;
        }

        if (commandType == null) {
            return -1;
        }

        for (int i = 0; i < commands.length; i++) {
            if (commands[i] != null && commandType.isAssignableFrom(commands[i].getClass())) {
                return i;
            }
        }

        return -1;
    }

    /**
     * Finds and returns the {@link Command} at the given <code>index</code> of the {@link Command} list.
     * @param index
     * @return
     */
    public Command getCommand(int index) {
        if (index < 0 || index > commands.length) {
            throw new IllegalArgumentException();
        }

        return commands[index];
    }

    /**
     * <p>Add a {@link Command} at the given <code>index</code> of the list of 
     * {@link Command}s that will be called in turn when this {@link Chain}'s 
     * <code>execute()</code> method is called.
     * Once <code>execute()</code> has been called
     * at least once, it is no longer possible to add additional
     * {@link Command}s; instead, an exception will be thrown.</p>

     * @param index The index at which the command should be inserted
     * @param command The {@link Command} to be added
     * @exception IllegalArgumentException if <code>command</code>
     *  is <code>null</code>, or 
     *  <code>index</code> is negative or greater than the size of the command list.
     * @exception IllegalStateException if this {@link Chain} has already
     *  been executed at least once, so no further configuration is allowed
     */
    public void addCommand(int index, Command command) {
        if (index < 0 || index > commands.length) {
            throw new IllegalArgumentException();
        }

        if (command == null) {
            throw new IllegalArgumentException();
        }

        if (frozen) {
            throw new IllegalStateException();
        }

        Command[] results = new Command[commands.length + 1];
        System.arraycopy(commands, 0, results, 0, index);
        System.arraycopy(commands, index, results, index + 1, commands.length - index);
        results[index] = command;
        commands = results;
    }

    /**
     * <p>Remove the {@link Command} at the given <code>index</code> of the list of 
     * {@link Command}s that will be called in turn when this {@link Chain}'s 
     * <code>execute()</code> method is called.
     * Once <code>execute()</code> has been called
     * at least once, it is no longer possible to add additional
     * {@link Command}s; instead, an exception will be thrown.</p>

     * @param index The index to add in the command list
     * @return true if successfully removed.
     * @exception IllegalArgumentException if <code>index</code> is out of bound.
     * @exception IllegalStateException if this {@link Chain} has already
     *  been executed at least once, so no further configuration is allowed
     */
    public boolean removeCommand(int index) {
        if (index < 0 || index > commands.length) {
            throw new IllegalArgumentException();
        }

        if (frozen) {
            throw new IllegalStateException();
        }

        Command[] results = new Command[commands.length - 1];
        System.arraycopy(commands, 0, results, 0, index);
        System.arraycopy(commands, index + 1, results, index, commands.length - index - 1);
        commands = results;

        return true;
    }

    /**
     * <p>Remove the {@link Command} from the list of 
     * {@link Command}s that will be called in turn when this {@link Chain}'s 
     * <code>execute()</code> method is called.
     * Once <code>execute()</code> has been called
     * at least once, it is no longer possible to add additional
     * {@link Command}s; instead, an exception will be thrown.</p>

     * @param command The {@link Command} to be added
     * @return true if successfully removed.
     * @exception IllegalArgumentException if <code>index</code> is out of bound.
     * @exception IllegalStateException if this {@link Chain} has already
     *  been executed at least once, so no further configuration is allowed
     */
    public boolean removeCommand(Command command) {
        if (command == null) {
            throw new IllegalArgumentException();
        }

        int index = getCommandIndex(command);

        if (index != -1) {
            return removeCommand(index);
        }

        return false;
    }

    /**
     * {@inheritDoc}
     * <p>
     * <em>Note:</em> The method implementation takes the strategy to catch all the exceptions
     * thrown while executing commands in the chain in order to proceed to the next commands
     * even when a command throws an exception in an earlier step.
     * This guarantees each of the internal commands are always
     * to execute. So, for example, a command, which is responsible for cleaning up all the
     * resources used during the reverse proxy processing, will always have a chance to do
     * its job in this reverse proxy processing chain.
     * </p>
     */
    @Override
    public boolean execute(Context context) throws ReverseProxyException, IOException
    {
        try 
        {
            super.execute(context);
        }
        catch (Exception e)
        {
            ((ProxyContext) context).addException(e);
        }

        return false;
    }

    /**
     * Return a list including all the descendant {@link Command}s,
     * type of which is <code>AbstractProxyCommand</code>
     * in this proxy processing chain.
     * @return
     */
    public List<AbstractProxyCommand> getAllProxyCommands()
    {
        List<AbstractProxyCommand> allCommands = new LinkedList<AbstractProxyCommand>();
        findDescendantProxyCommands(allCommands);
        return allCommands;
    }

    private void findDescendantProxyCommands(List<AbstractProxyCommand> descendantProxyCommands)
    {
        if (commands != null)
        {
            for (Command command : commands)
            {
                if (command instanceof AbstractProxyCommand)
                {
                    descendantProxyCommands.add((AbstractProxyCommand) command);
                }
                else if (command instanceof ProxyProcessingChain)
                {
                    ((ProxyProcessingChain) command).findDescendantProxyCommands(descendantProxyCommands);
                }
            }
        }
    }
}
